#include <stdlib.h>
#include <memory.h>
#include <dnpap.h>
#include <config.h>
#include <message.h>
#include <mac.h>
#include <hash.h>
#include <sys.h>
#include <protocol.h>
#include <dtopn.h>
#include <dfilter.h>

#include "dhoste.h"
#include "dhostx.h"
#include "dhostc.h"


static WORD HostMaxNrHosts = 2000;


static VOID HostCallback(MAC_COLL *collector, PROT_PKT *pkt);

static BOOLEAN HostAddLexList(DNPAP_HOST_CONTROL *hostcontrol, DNPAP_HOST *host);
static BOOLEAN HostRemoveLexList(DNPAP_HOST_CONTROL *hostcontrol, DNPAP_HOST *host);

static BOOLEAN HostAddTimeList(DNPAP_HOST_CONTROL *hostcontrol, DNPAP_HOST *host);
static BOOLEAN HostRemoveTimeList(DNPAP_HOST_CONTROL *hostcontrol, DNPAP_HOST *host);

static DNPAP_HOST* HostAddLRUList(DNPAP_HOST_CONTROL *hostcontrol, DNPAP_HOST *host);
static BOOLEAN HostUpdateLRUList(DNPAP_HOST_CONTROL *hostcontrol, DNPAP_HOST *host);
static BOOLEAN HostRemoveLRUList(DNPAP_HOST_CONTROL *hostcontrol, DNPAP_HOST *host);

static BOOLEAN HostUpdateTimeTable(DNPAP_HOST_CONTROL *hostcontrol);

static VOID DelHosts(DNPAP_HOST_CONTROL *hostcontrol);



BOOLEAN DnpapHostConfigInit(VOID)
{
	ConfigGetShort("beholder.dnpaphost.maxnrhosts", &HostMaxNrHosts);

	if (HostMaxNrHosts < 2)
    {
        DnpapMessage(DMC_WARNING, DHOST_MAX, "dhostcontrol: beholder.host.maxnrhosts < 2, setting it to 2");
        HostMaxNrHosts = 2;
    }

	return TRUE;
}


BOOLEAN HostCInit(DNPAP_HOST_CONTROL *hostcontrol)
{
LONG source[] = {1,3,6,1,2,1,2,2,1,1,1};

    hostcontrol->Index = 0;
    memcpy(hostcontrol->Source, source, sizeof(source));
    hostcontrol->SourceLen = sizeof(source)/sizeof(source[0]);
    hostcontrol->TableSize = 0;
    hostcontrol->LastDeleteTime = 0;
    hostcontrol->Owner[0] = '\0';
    hostcontrol->OwnerLen = 0;
    hostcontrol->Status = SNMP_INVALID;
    
    hostcontrol->AcceptType = 1;
    hostcontrol->Matches = 0;
    hostcontrol->NrBuckets = 0;

    if ((hostcontrol->Iface =
        MacIfaceGet((WORD) hostcontrol->Source[hostcontrol->SourceLen-1])) == NULL)
    {
        DnpapMessage(DMC_ERROR, DHOST_NETINIT, "dhostcontrol: network initialisation failed");
        return (FALSE);
    }
	
    hostcontrol->Table = NULL;
    
    hostcontrol->LexList = NULL;
    
    hostcontrol->TimeList = NULL;
    hostcontrol->TimeLast = NULL;
    hostcontrol->TimeTable = NULL;
    hostcontrol->TimeListUpdated = FALSE;

	hostcontrol->LRUList = NULL;
	hostcontrol->LRULast = NULL;
	
	return TRUE;
}


BOOLEAN HostCStart(DNPAP_HOST_CONTROL *hostcontrol)
{
    hostcontrol->Coll.Rcve       = HostCallback;
    hostcontrol->Coll.specific   = hostcontrol;
    if (!MacCollRegister(&(hostcontrol->Coll)))
    {
        DnpapMessage(DMC_ERROR, DHOST_NETERR, "dhostcontrol: network initialisation failed");
        return FALSE;
    }

    if ((hostcontrol->Table = NewHash(2011, NULL)) == NULL)
    {
        DnpapMessage(DMC_ERROR, DHOST_HASHERR, "dhostcontrol: can not create hashtable");
        return FALSE;
    }

    if ((hostcontrol->TimeTable = (DNPAP_HOST**)DnpapMalloc(HostMaxNrHosts*sizeof(DNPAP_HOST*))) == NULL)
    {
        DnpapMessage(DMC_ERROR, DHOST_TIMER, "dhostcontrol: can not create time ordered table");
        return FALSE;
    }

    return TRUE;
}


BOOLEAN HostCStop(DNPAP_HOST_CONTROL *hostcontrol)
{
    MacCollRemove(&(hostcontrol->Coll));

    DelHash(hostcontrol->Table);

    DelHosts(hostcontrol);

    DnpapFree(hostcontrol->TimeTable);

    return TRUE;
}


VOID HostCallback(MAC_COLL *collector, PROT_PKT *pkt)
{
DNPAP_HOST_CONTROL *hostcontrol = collector->specific;
DNPAP_HOST *host = NULL, *oldhost = NULL;
BYTE broadcast[HOST_SZEADDR] = {0xff,0xff,0xff,0xff,0xff,0xff};
PROT_OBJ Interface = {0, {1,2}};
PROT_OBJ Time = {0, {1,6}};
PROT_OBJ Size = {0, {1,4}};
PROT_OBJ Dst = {1, {2,1}};
PROT_OBJ Src = {1, {2,2}};
PROT_OBJ DstTCP = {3, {4,2}};
PROT_OBJ SrcTCP = {3, {4,1}};
ULONG size;
ULONG time;
LONG iat, iatindex, ist, istindex;
BOOLEAN hostcreated, filter;

    filter = DnpapFilterPkt(0, hostcontrol->Index, 0, pkt);
    if ((hostcontrol->AcceptType == 1 && filter == TRUE) ||
        (hostcontrol->AcceptType == 2 && filter == FALSE))
        hostcontrol->Matches++;
    else
        return;

    if (ProtGetField(pkt,&Interface) == TRUE &&
        Interface.Syntax.LngInt == hostcontrol->Source[hostcontrol->SourceLen-1])
    {
        if (ProtGetField(pkt,&Time) == TRUE &&
            ProtGetField(pkt,&Size) == TRUE &&
            ProtGetField(pkt,&Src) == TRUE &&
            ProtGetField(pkt,&Dst) == TRUE)
        {
            time = Time.Syntax.LngUns;
            size = Size.Syntax.LngUns + 4L;
            if (Src.SyntaxLen != HOST_SZEADDR || Dst.SyntaxLen != HOST_SZEADDR)
            {
                DnpapMessage(DMC_ERROR, DHOST_INVPACKET, "dhostcontrol: invalid packet");
                return;
            }

            hostcreated = FALSE;

            if ((host = HashSearch(hostcontrol->Table, Src.Syntax.BufChr, HOST_SZEADDR)) == NULL)
            {
                /*  first try to add the new host  */
                if ((host = DnpapMalloc(sizeof(DNPAP_HOST))) != NULL)
                {
                    memset(host, 0, sizeof(DNPAP_HOST));
                    memcpy(host->Address, Src.Syntax.BufChr, HOST_SZEADDR);
                    if ((host->IATBuckets = DnpapMalloc(hostcontrol->NrBuckets*sizeof(host->IATBuckets[0]))) == NULL)
                    {
                        DnpapMessage(DMC_WARNING, DHOST_NIAT, "dhostcontrol: IAT buckets could not be added to a host");
                        DnpapFree(host);
                        host = NULL;
                    }
                    memset(host->IATBuckets, 0, (size_t)hostcontrol->NrBuckets*sizeof(host->IATBuckets[0]));
                    if ((host->ISTBuckets = DnpapMalloc(hostcontrol->NrBuckets*sizeof(host->ISTBuckets[0]))) == NULL)
                    {
                        DnpapMessage(DMC_WARNING, DHOST_NIST, "dhostcontrol: IST buckets could not be added to a host");
                        DnpapFree(host->IATBuckets);
                        DnpapFree(host);
                        host = NULL;
                    }
                    memset(host->ISTBuckets, 0, (size_t)hostcontrol->NrBuckets*sizeof(host->ISTBuckets[0]));
                    if (HashAdd(hostcontrol->Table, host->Address, HOST_SZEADDR, host) == NULL)
                    {
                        DnpapMessage(DMC_WARNING, DHOST_NADD, "dhostcontrol: host could not be added to the hash table");
                        DnpapFree(host);
                        host = NULL;
                    }
                    else
                    {
                        hostcreated = TRUE;

                        HostAddLexList(hostcontrol, host);
                        HostAddTimeList(hostcontrol, host);
                        oldhost = HostAddLRUList(hostcontrol, host);
                        hostcontrol->TableSize++;

                        if (hostcontrol->TableSize > (LONG)HostMaxNrHosts)
                        {
                            HostRemoveLRUList(hostcontrol, oldhost);
                            HostRemoveTimeList(hostcontrol, oldhost);
                            HostRemoveLexList(hostcontrol, oldhost);
                            HashRemove(hostcontrol->Table, oldhost->Address, HOST_SZEADDR);
                            DnpapTopNEntryReset(hostcontrol->Index, oldhost);
                            DnpapFree(oldhost);
                            hostcontrol->LastDeleteTime = SysTime();
                            hostcontrol->TableSize--;
                        }
                    }
                }
            }
            if (host != NULL)
            {
                if (hostcreated == TRUE)
                {
                    host->ISTTime = time;
                    host->ISTBuckets[0]++;
                }
                if (hostcontrol->NrBuckets > 0)
                {
                    if (hostcreated == TRUE)
                    {
                        host->ISTTime = time;
                        host->ISTBuckets[0]++;
                    }
                    else
                    {
                        ist = time - host->ISTTime;
                        host->ISTTime = time;
                        istindex = ist/hostcontrol->Interval;
                        if (istindex >= hostcontrol->NrBuckets)
                            istindex = hostcontrol->NrBuckets-1;
                        host->ISTBuckets[istindex]++;
                    }
                }

                host->OutPkts++;
                host->OutOctets += size;
                if (!memcmp(Dst.Syntax.BufChr, &broadcast, HOST_SZEADDR))
                    host->OutBroadcastPkts++;
                else
                    if (Dst.Syntax.BufChr[0] & 0x01)
                        host->OutMulticastPkts++;
                        
		        if (size == 64)
		            host->OutBucket1Pkts++;
		        if (size >= 65)
		        {
		            if (size <= 127)
		                host->OutBucket2Pkts++;
		            else if (size <= 255)
		                host->OutBucket3Pkts++;
		            else if (size <= 511)
		                host->OutBucket4Pkts++;
		            else if (size <= 1023)
		                host->OutBucket5Pkts++;
		            else if (size <= 1518)
		                host->OutBucket6Pkts++;
		        }

		        if (ProtGetField(pkt, &SrcTCP) == TRUE)
		            host->OutTCPPkts++;
		        else
		            host->OutOtherPkts++;

                HostUpdateLRUList(hostcontrol, host);
                DnpapTopNEntryUpdate(hostcontrol->Index, host);
            }


            hostcreated = FALSE;

            if ((host = HashSearch(hostcontrol->Table, Dst.Syntax.BufChr, HOST_SZEADDR)) == NULL)
            {
                if ((host = DnpapMalloc(sizeof(DNPAP_HOST))) != NULL)
                {
                    memset(host, 0, sizeof(DNPAP_HOST));
                    memcpy(host->Address, Dst.Syntax.BufChr, HOST_SZEADDR);
                    if ((host->IATBuckets = DnpapMalloc(hostcontrol->NrBuckets*sizeof(host->IATBuckets[0]))) == NULL)
                    {
                        DnpapMessage(DMC_WARNING, DHOST_NIAT, "dhostcontrol: IAT buckets could not be added to a host");
                        DnpapFree(host);
                        host = NULL;
                    }
                    memset(host->IATBuckets, 0, (size_t)hostcontrol->NrBuckets*sizeof(host->IATBuckets[0]));
                    if ((host->ISTBuckets = DnpapMalloc(hostcontrol->NrBuckets*sizeof(host->ISTBuckets[0]))) == NULL)
                    {
                        DnpapMessage(DMC_WARNING, DHOST_NIST, "dhostcontrol: IST buckets could not be added to a host");
                        DnpapFree(host->IATBuckets);
                        DnpapFree(host);
                        host = NULL;
                    }
                    memset(host->ISTBuckets, 0, (size_t)hostcontrol->NrBuckets*sizeof(host->ISTBuckets[0]));
                    if (HashAdd(hostcontrol->Table, host->Address, HOST_SZEADDR, host) == NULL)
                    {
                        DnpapMessage(DMC_WARNING, DHOST_NADD, "dhostcontrol: host could not be added to the hash table");
                        DnpapFree(host);
                        host = NULL;
                    }
                    else
                    {
                        hostcreated = TRUE;

                        HostAddLexList(hostcontrol, host);
                        HostAddTimeList(hostcontrol, host);
                        oldhost = HostAddLRUList(hostcontrol, host);
                        hostcontrol->TableSize++;

                        if (hostcontrol->TableSize > (LONG)HostMaxNrHosts)
                        {
                            HostRemoveLRUList(hostcontrol, oldhost);
                            HostRemoveTimeList(hostcontrol, oldhost);
                            HostRemoveLexList(hostcontrol, oldhost);
                            HashRemove(hostcontrol->Table, oldhost->Address, HOST_SZEADDR);
                            DnpapTopNEntryReset(hostcontrol->Index, oldhost);
                            DnpapFree(oldhost);
                            hostcontrol->LastDeleteTime = SysTime();
                            hostcontrol->TableSize--;
                        }
                    }
                }
            }
            if (host != NULL)
            {
                if (hostcontrol->NrBuckets > 0)
                {
                    if (hostcreated == TRUE)
                    {
                        host->IATTime = time;
                        host->IATBuckets[0]++;
                    }
                    else
                    {
                        iat = time - host->IATTime;
                        host->IATTime = time;
                        iatindex = iat/hostcontrol->Interval;
                        if (iatindex >= hostcontrol->NrBuckets)
                            iatindex = hostcontrol->NrBuckets-1;
                        host->IATBuckets[iatindex]++;
                    }
                }

                host->InPkts++;
                host->InOctets += size;
                
		        if (size == 64)
		            host->InBucket1Pkts++;
		        if (size >= 65)
		        {
		            if (size <= 127)
		                host->InBucket2Pkts++;
		            else if (size <= 255)
		                host->InBucket3Pkts++;
		            else if (size <= 511)
		                host->InBucket4Pkts++;
		            else if (size <= 1023)
		                host->InBucket5Pkts++;
		            else if (size <= 1518)
		                host->InBucket6Pkts++;
		        }

		        if (ProtGetField(pkt, &DstTCP) == TRUE)
		            host->InTCPPkts++;
		        else
		            host->InOtherPkts++;

                HostUpdateLRUList(hostcontrol, host);
                DnpapTopNEntryUpdate(hostcontrol->Index, host);
            }
        }
    }

    return;
}


BOOLEAN HostAddLexList(DNPAP_HOST_CONTROL *hostcontrol, DNPAP_HOST *host)
{
DNPAP_HOST *p, *q;

    if (hostcontrol->LexList == NULL)
    {
        hostcontrol->LexList = host;
        host->LexPrev = NULL;
        host->LexNext = NULL;
        return TRUE;
    }

    for (p = hostcontrol->LexList, q = NULL; p != NULL; q = p, p = p->LexNext)
    {
        if (memcmp(p->Address, host->Address, HOST_SZEADDR) >= 0)
            break;
    }

    if (q != NULL)
        q->LexNext = host;
    else
        hostcontrol->LexList = host;

    if (p != NULL)
        p->LexPrev = host;

    host->LexPrev = q;
    host->LexNext = p;

    return TRUE;
}


BOOLEAN HostAddTimeList(DNPAP_HOST_CONTROL *hostcontrol, DNPAP_HOST *host)
{
    if (hostcontrol->TimeList == NULL)
    {
        hostcontrol->TimeList = host;
        hostcontrol->TimeLast = host;
        host->TimePrev = NULL;
        host->TimeNext = NULL;
        
	    hostcontrol->TimeListUpdated = TRUE;

        return TRUE;
    }

    host->TimePrev = hostcontrol->TimeLast;
    host->TimeNext = NULL;
    hostcontrol->TimeLast->TimeNext = host;
    hostcontrol->TimeLast = host;
    
	hostcontrol->TimeListUpdated = TRUE;
	
    return TRUE;
}


DNPAP_HOST* HostAddLRUList(DNPAP_HOST_CONTROL *hostcontrol, DNPAP_HOST *host)
{
    if (hostcontrol->LRUList == NULL)
    {
        hostcontrol->LRUList = host;
        hostcontrol->LRULast = host;
        host->LRUPrev = NULL;
        host->LRUNext = NULL;
        
        return NULL;
    }

    host->LRUNext = hostcontrol->LRUList;
    host->LRUPrev = NULL;
    hostcontrol->LRUList->LRUPrev = host;
    hostcontrol->LRUList = host;

    return hostcontrol->LRULast;
}      


BOOLEAN HostUpdateLRUList(DNPAP_HOST_CONTROL *hostcontrol, DNPAP_HOST *host)
{
    if (hostcontrol->TableSize > 1 && hostcontrol->LRUList != host)
    {
        HostRemoveLRUList(hostcontrol, host);
        HostAddLRUList(hostcontrol, host);
    }
    return TRUE;
}


BOOLEAN HostRemoveLexList(DNPAP_HOST_CONTROL *hostcontrol, DNPAP_HOST *host)
{
    if (host->LexPrev != NULL)
        host->LexPrev->LexNext = host->LexNext;
    else
        hostcontrol->LexList = host->LexNext;
    if (host->LexNext != NULL)
        host->LexNext->LexPrev = host->LexPrev;

    host->LexPrev = NULL;
    host->LexNext = NULL;
    
    return TRUE;
}


BOOLEAN HostRemoveTimeList(DNPAP_HOST_CONTROL *hostcontrol, DNPAP_HOST *host)
{
    if (host->TimePrev != NULL)
        host->TimePrev->TimeNext = host->TimeNext;
    else
        hostcontrol->TimeList = host->TimeNext;
    if (host->TimeNext != NULL)
        host->TimeNext->TimePrev = host->TimePrev;
    else
        hostcontrol->TimeLast = host->TimePrev;

	hostcontrol->TimeListUpdated = TRUE;
	
    host->TimePrev = NULL;
    host->TimeNext = NULL;
    
    return TRUE;
}


BOOLEAN HostRemoveLRUList(DNPAP_HOST_CONTROL *hostcontrol, DNPAP_HOST *host)
{
    if (host->LRUPrev != NULL)
        host->LRUPrev->LRUNext = host->LRUNext;
    else
        hostcontrol->LRUList = host->LRUNext;
    if (host->LRUNext != NULL)
        host->LRUNext->LRUPrev = host->LRUPrev;
    else
        hostcontrol->LRULast = host->LRUPrev;

    host->LRUPrev = NULL;
    host->LRUNext = NULL;
    
    return TRUE;
}


VOID DelHosts(DNPAP_HOST_CONTROL *hostcontrol)
{
DNPAP_HOST *host1, *host2;

    for (host2 = hostcontrol->LexList; host2 != NULL; host2 = host1)
    {
        host1 = host2->LexNext;
        DnpapFree(host2);
        hostcontrol->TableSize--;
    }
    hostcontrol->LexList = NULL;
}


DNPAP_HOST* HostHashSearch(DNPAP_HOST_CONTROL *hostcontrol, DNPAP_HOST *host, WORD len)
{
DNPAP_HOST* p;

    if (len == HOST_SZEADDR &&
        (p = HashSearch(hostcontrol->Table, host->Address, HOST_SZEADDR)) != NULL)
        return p;
    else
    {
        for (p = hostcontrol->LexList; p != NULL; p = p->LexNext)
        {
            if (memcmp(p->Address, host->Address, len) >= 0)
                break;
        }
        return p;
    }
    return NULL;
}


DNPAP_HOST* HostHashSearch2(DNPAP_HOST_CONTROL *hostcontrol, SNMP_OBJECT *obj, WORD idlen)
{
WORD i;
BYTE address[HOST_SZEADDR];

    for (i = 0; i < HOST_SZEADDR; i++)
        address[i] = (BYTE)obj->Id[idlen+1+i];
    return HashSearch(hostcontrol->Table, address, HOST_SZEADDR);
}


DNPAP_HOST* HostTimeSearch(DNPAP_HOST_CONTROL *hostcontrol, SNMP_OBJECT *obj, WORD idlen)
{
	if (hostcontrol->TimeListUpdated == TRUE)
		HostUpdateTimeTable(hostcontrol);
    return hostcontrol->TimeTable[obj->Id[idlen+1]-1];
}


LONG HostSearchCreationOrder(DNPAP_HOST_CONTROL *hostcontrol, DNPAP_HOST *host)
{
	if (hostcontrol->TimeListUpdated == TRUE)
		HostUpdateTimeTable(hostcontrol);
    return host->TimeListPtr - hostcontrol->TimeTable + 1;
}


BOOLEAN HostUpdateTimeTable(DNPAP_HOST_CONTROL *hostcontrol)
{
LONG i;
DNPAP_HOST *p;

	for (p = hostcontrol->TimeList, i = 0; p != NULL; p = p->TimeNext, i++)
	{
		hostcontrol->TimeTable[i] = p;
		p->TimeListPtr = hostcontrol->TimeTable+i;
	}
	
	if (i != hostcontrol->TableSize)
	{
		DnpapMessage(DMC_FATAL, DHOST_UPDTIME, "dhostcontrol: internal error in time list and table");
		DnpapExit(DHOST_UPDTIME);
	}
	
	hostcontrol->TimeListUpdated = FALSE;
	
	return TRUE;
}

