/************************************************************************
** MODULE INFORMATION*
**********************
**     FILE     NAME:       vipscroll.c
**     SYSTEM   NAME:       VIP
**     ORIGINAL AUTHOR(S):  Alfred Kayser
**     VERSION  NUMBER:     1.00
**     CREATION DATE:       1992/5/29
**
** DESCRIPTION: Empty Button Module.
**              
*************************************************************************
** CHANGES INFORMATION **
*************************
** REVISION:    $Revision$
** WORKFILE:    $Workfile$
** LOGINFO:     $Log$
*************************************************************************/
#include "vipinc.h"

#define SCROLSIZE 16

/* Down and right increase the current value */
/* Up and left decrease the current value */
#define DOWNBUT(w)  SCROLLDATA(w, b1)
#define UPBUT(w)    SCROLLDATA(w, b2)
#define MOVEBUT(w)  SCROLLDATA(w, b3)
#define LEFTBUT(w)  UPBUT(w)
#define RIGHTBUT(w) DOWNBUT(w)

BYTE upImage[] =
{
    0x00, 0x00, 0x00, 0x00, 0x06, 0x00, 0x0f, 0x00,
    0x1f, 0x80, 0x3f, 0xc0, 0x7f, 0xe0, 0x0f, 0x00,
    0x0f, 0x00, 0x0f, 0x00, 0x00, 0x00, 0x00, 0x00,

};
BYTE downImage[] =
{
    0x00, 0x00, 0x00, 0x00, 0x0f, 0x00, 0x0f, 0x00,
    0x0f, 0x00, 0x7f, 0xe0, 0x3f, 0xc0, 0x1f, 0x80,
    0x0f, 0x00, 0x06, 0x00, 0x00, 0x00, 0x00, 0x00,
};

BYTE rightImage[] =
{
    0x00, 0x00, 0x04, 0x00, 0x06, 0x00, 0x07, 0x00,
    0x3f, 0x80, 0x3f, 0xc0, 0x3f, 0xc0, 0x3f, 0x80,
    0x07, 0x00, 0x06, 0x00, 0x04, 0x00, 0x00, 0x00,

};
BYTE leftImage[] =
{
    0x00, 0x00, 0x02, 0x00, 0x06, 0x00, 0x0e, 0x00,
    0x1f, 0xc0, 0x3f, 0xc0, 0x3f, 0xc0, 0x1f, 0xc0,
    0x0e, 0x00, 0x06, 0x00, 0x02, 0x00, 0x00, 0x00,
};

PRIVAT VIPINFO *VipOpenScroll(VIPINFO *parent, int x, int y, int w, int h);
PRIVAT LONG ScrollHandler(VIPINFO *wip, USHORT msg, MPARAM mp1, MPARAM mp2);
PRIVAT void ScrollUpdate(VIPINFO *wip, HPS hps, BOOLEAN all);
PRIVAT BOOLEAN ScrollAdjust(VIPINFO *wip, SWP *pswp);
PRIVAT VOID PushButton(VIPINFO *wip, HPS hps, RECTL *rct);
PRIVAT VOID ScrollSetSize(VIPINFO *wip);
PRIVAT VOID ScrollSetMove(VIPINFO *wip);

IMPORT LONG vipDesktopH, vipDesktopW;
IMPORT HAB vipHab;

#define SVERT 0x0001
#define SHORZ 0x0002

/**************************************************************
** NAME:        VipOpenVScroll                            [API]
** SYNOPSIS:    VIPINFO *VipOpenVScroll(VIPINFO *parent,
**                  int x, int y, int w, int h)
** DESCRIPTION: Opens a vertical scrollbar window.
** RETURNS:     VIPINFO pointer
**************************************************************/
EXPORT VIPINFO *
VipOpenVScroll(VIPINFO *parent, int x, int y, int w, int h)
{
    VIPINFO *wip;

    wip = VipOpenScroll(parent, x,y,w,h);
    if (wip) SCROLLDATA(wip,flags) = SVERT;
    return wip;
}

/**************************************************************
** NAME:        VipOpenHScroll                            [API]
** SYNOPSIS:    VIPINFO *VipOpenHScroll(VIPINFO *parent,
**                  int x, int y, int w, int h)
** DESCRIPTION: Opens a horizontal scrollbar window.
** RETURNS:     VIPINFO pointer
**************************************************************/
EXPORT VIPINFO *
VipOpenHScroll(VIPINFO *parent, int x, int y, int w, int h)
{
    VIPINFO *wip;

    wip = VipOpenScroll(parent, x,y,w,h);
    if (wip) SCROLLDATA(wip,flags) = SHORZ;
    return wip;
}


/**************************************************************
** NAME:        VipOpenScroll 
** SYNOPSIS:    VIPINFO *VipOpenScroll(VIPINFO *parent,
**                  int x, int y, int w, int h)
** DESCRIPTION: Opens a  scroll window.
** RETURNS:     VIPINFO pointer
**************************************************************/
PRIVAT VIPINFO *
VipOpenScroll(VIPINFO *parent, int x, int y, int w, int h)
{
    VIPINFO *wip;
	struct _scrolldata *data;

    if (!(data = VipMalloc(sizeof(struct _scrolldata))))
        return NULL;
    if (!(wip = VipOpenSimple(parent, x,y,w,h)))
    {
        VipFree(data);
        return NULL;
    }
    data->min   = 0;
    data->max   = 10;
    data->size  = 1;
    data->current = 0;
    data->range = data->max - data->min;
    data->flags = SVERT;

    wip->handler = ScrollHandler;
    wip->update = ScrollUpdate;
    wip->adjust = ScrollAdjust;
    wip->type = T_SCROLL;
    wip->scrolldata = data;
    wip->border = 1;
    wip->btype = VIP_LOW;

    return wip;
}


/**************************************************************
** NAME:        VipSetScrollCurrent                       [API]
** SYNOPSIS:    void VipSetScrollCurrent(VIPINFO *wip,
**                    LONG current)
** DESCRIPTION: Sets the current value of the scrollbar.
**              As with all VipSet functions the corresponding
**              view is not updated until a 'VipUpdate' or
**              'VipShow' forces a redraw or refresh.
** RETURNS:     void
**************************************************************/
void
VipSetScrollCurrent(VIPINFO *wip, LONG current)
{
    TYPETEST(wip,T_SCROLL,return);
    SCROLLDATA(wip,current)=current;
    if (wip->callback)
        wip->callback(wip, wip->pointer, (int)SCROLLDATA(wip,current));
}

                           
/**************************************************************
** NAME:        VipSetScrollSize                       [API]
** SYNOPSIS:    void VipSetScrollSize(VIPINFO *wip,
**                    LONG size)
** DESCRIPTION: Sets size of the moving part of the scrollbar.
**              As with all VipSet functions the corresponding
**              view is not updated until a 'VipUpdate' or
**              'VipShow' forces a redraw or refresh.
** RETURNS:     void
**************************************************************/
void
VipSetScrollSize(VIPINFO *wip, LONG size)
{
    TYPETEST(wip,T_SCROLL,return);
    if (size>SCROLLDATA(wip,range))
        size=SCROLLDATA(wip,range);
    SCROLLDATA(wip,size)=size;
    SREDRAW(wip);
}


/**************************************************************
** NAME:        VipSetScrollRange                         [API]
** SYNOPSIS:    void VipSetScrollRange(VIPINFO *wip,
**                  LONG min, LONG max)
** DESCRIPTION: Sets the min and max value of the scrollbar
** RETURNS:     void
**************************************************************/
void 
VipSetScrollRange(VIPINFO *wip, LONG min, LONG max)
{
    TYPETEST(wip,T_SCROLL,return);
    SCROLLDATA(wip,min)=min;
    SCROLLDATA(wip,max)=max;
    SCROLLDATA(wip,range)=max-min;
    if (SCROLLDATA(wip, size)>SCROLLDATA(wip,range))
        SCROLLDATA(wip, size)=SCROLLDATA(wip,range);
    SREDRAW(wip);
}


/**************************************************************
** NAME:        VipSetScrollCallBack                      [API]
** SYNOPSIS:    void VipSetScrollCallBack(VIPINFO *wip,
**                  VIP_SCROLLCALL callback, void *ptr)
** DESCRIPTION: Installs a callback function. This function
**              will be called when the button is buttoned
**              or released. The function must have the
**              following syntax:
**              void callback(VIPINFO *wip, VOID *ptr, int c);
**              The pointer <ptr> can point to anything, the
**              argument <f> will be true when the button is
**              pressed.
** RETURNS:     void
**************************************************************/
EXPORT void
VipSetScrollCallBack(VIPINFO *wip, VIP_SCROLLCALL callback, void *ptr)
{
    TYPETEST(wip,T_SCROLL,return);
    wip->callback = (VIP_CALLBACK)callback;
    wip->pointer = ptr;
}

#define NOP 0
#define UP 1
#define DOWN 2
#define MOVING 3

PRIVAT LONG ScrollHandler(VIPINFO *wip, USHORT msg, MPARAM mp1, MPARAM mp2)
{
    static int active=FALSE;
    static int action=NOP;
    static LONG offset;
    LONG s, sl;
    HPS hps;
    POINTL pp;
    int newAction;

	switch(msg)
	{
    case WM_BUTTON1DOWN:
        WinSetFocus(HWND_DESKTOP, wip->win);
        WinSetCapture(HWND_DESKTOP, wip->win);
        active=TRUE;
        action=NOP;
        /* FALLTHROUGH */

    case WM_MOUSEMOVE:
        if (!active) break;
        pp.x=SHORT1FROMMP(mp1);
        pp.y=SHORT2FROMMP(mp1);

        newAction=NOP;
        if (action==MOVING)
            newAction=MOVING;
        else if (WinPtInRect(vipHab, &DOWNBUT(wip), &pp))
            newAction=DOWN;
        else if (WinPtInRect(vipHab, &UPBUT(wip), &pp))
            newAction=UP;
        else if (pp.x>=MOVEBUT(wip).xLeft && pp.x<=MOVEBUT(wip).xRight
              && pp.y>DOWNBUT(wip).yTop&& pp.y<UPBUT(wip).yBottom)
            newAction=MOVING;
        hps=WinGetPS(wip->win);
        if (newAction!=action)
        {
            switch(action)
            {
            case DOWN:
                VipDrawBorder(wip, &DOWNBUT(wip), hps, VIP_RISE, 2);
                break;
            case UP:
                VipDrawBorder(wip, &UPBUT(wip), hps, VIP_RISE, 2);
                break;
            case MOVING:
                VipDrawBorder(wip, &MOVEBUT(wip), hps, VIP_RISE, 2);
                break;
            }
            action=newAction;
            switch(action)
            {
            case DOWN:
                PushButton(wip, hps, &DOWNBUT(wip));
                break;
            case UP:
                PushButton(wip, hps, &UPBUT(wip));
                break;
            case MOVING:
                PushButton(wip, hps, &MOVEBUT(wip));
                offset=pp.y;
                break;
            }
        }
        switch(action)
        {
        case UP: break;
        case DOWN: break;
        case MOVING:
            if (pp.y>=0x8000L) pp.y=0;  /* Bug in PM about unsigned coord's */
            if ((SCROLLDATA(wip,flags)&SVERT)
             && (SCROLLDATA(wip,size)<SCROLLDATA(wip,range)))
            {
                if (pp.y<DOWNBUT(wip).yTop) pp.y=DOWNBUT(wip).yTop;
                if (pp.y>UPBUT(wip).yBottom) pp.y=UPBUT(wip).yBottom;
                if (pp.y - offset)
                {
                    VipDrawBorder(wip, &MOVEBUT(wip), hps, VIP_NONE, 2);
                    sl = MOVEBUT(wip).yTop - MOVEBUT(wip).yBottom;
                    MOVEBUT(wip).yTop += pp.y - offset;
                    if (MOVEBUT(wip).yTop > UPBUT(wip).yBottom)
                        MOVEBUT(wip).yTop = UPBUT(wip).yBottom;
                    if (MOVEBUT(wip).yTop - sl < DOWNBUT(wip).yTop)
                        MOVEBUT(wip).yTop = DOWNBUT(wip).yTop + sl;
                    MOVEBUT(wip).yBottom = MOVEBUT(wip).yTop - sl;
                    s = (UPBUT(wip).yBottom - MOVEBUT(wip).yTop)*SCROLLDATA(wip,range);
                    SCROLLDATA(wip,current)= s / (UPBUT(wip).yBottom - DOWNBUT(wip).yTop - sl);
                    offset = pp.y;
                    VipDrawBorder(wip, &MOVEBUT(wip), hps, VIP_RISE, 1);
                    if (wip->callback)
                        wip->callback(wip, wip->pointer, (int)SCROLLDATA(wip,current));
                }
            }
            break;
        }
        WinReleasePS(hps);
        return TRUE;
    
    case WM_BUTTON1UP:
        WinSetCapture(HWND_DESKTOP, NULL);
        active=FALSE;
        hps=WinGetPS(wip->win);
        switch(action)
        {
        case UP:    VipDrawBorder(wip, &UPBUT(wip),   hps, VIP_RISE, 2);break;
        case DOWN:  VipDrawBorder(wip, &DOWNBUT(wip), hps, VIP_RISE, 2);break;
        case MOVING:VipDrawBorder(wip, &MOVEBUT(wip), hps, VIP_RISE, 2);break;
        }
        WinReleasePS(hps);
        action=NOP;
        return TRUE;

    case WM_SIZE:
        ScrollSetSize(wip);
        break;
	}
	return VipSimpleHandler(wip, msg, mp1, mp2);
}


PRIVAT BOOLEAN
ScrollAdjust(VIPINFO *wip, SWP *pswp)
{
    BOOLEAN changed=FALSE;
    SHORT w, h;

    if (SCROLLDATA(wip,flags)&SVERT)
    {
        w = SCROLSIZE+2*wip->border; 
        h = (SHORT)((wip->parent) ? wip->parent->cy : vipDesktopH);
    }
    else
    {
        w = (SHORT)((wip->parent) ? wip->parent->cx : vipDesktopW);
        h = SCROLSIZE+2*wip->border; 
    }

    if ((wip->w==0) && (pswp->cx!=w))
    {
        wip->cx=pswp->cx=w;
        changed=TRUE;
    }
    if ((wip->h==0) && (pswp->cy!=h))
    {
        wip->cy=pswp->cy=h;
        changed=TRUE;
    }
    return changed;
}


PRIVAT void
ScrollUpdate(VIPINFO *wip, HPS hps, BOOLEAN all)
{
    SIZEL size;
    POINTL pp;
    if (all || QREDRAW(wip))
    {
        ScrollSetSize(wip);
        RREDRAW(wip);
        VipBorder(wip, hps);
        VipDrawBorder(wip, &DOWNBUT(wip), hps, VIP_RISE, 2);
        VipDrawBorder(wip, &UPBUT(wip), hps, VIP_RISE, 2);
        VipDrawBorder(wip, &MOVEBUT(wip), hps, VIP_RISE, 2);
        size.cx=12;
        size.cy=12;
        GpiSetColor(hps, FOREGROUND(wip));
        if (SCROLLDATA(wip, flags)&SVERT)
        {
            LONG x = (UPBUT(wip).xLeft + UPBUT(wip).xRight)/2 - 6;
            MOVE(x ,UPBUT(wip).yBottom+13);
            GpiImage(hps, 0, &size, sizeof(upImage), upImage);
            MOVE(x,DOWNBUT(wip).yBottom+13);
            GpiImage(hps, 0, &size, sizeof(downImage), downImage);
        }
        else
        {
            LONG y = (LEFTBUT(wip).yBottom + LEFTBUT(wip).yTop)/2 + 5;
            MOVE(LEFTBUT(wip).xLeft+2, y);
            GpiImage(hps, 0, &size, sizeof(leftImage), leftImage);
            MOVE(RIGHTBUT(wip).xLeft+2, y);
            GpiImage(hps, 0, &size, sizeof(rightImage), rightImage);
        }
    }
    else
    {
        ScrollSetMove(wip);
        GpiSetColor(hps, BACKGROUND(wip));
        MOVE(DOWNBUT(wip).xLeft, DOWNBUT(wip).yTop);
        BLOCK(UPBUT(wip).xRight, UPBUT(wip).yBottom);
        VipDrawBorder(wip, &MOVEBUT(wip), hps, VIP_RISE, 2);
    }
}


PRIVAT VOID
PushButton(VIPINFO *wip, HPS hps, RECTL *rct)
{
    POINTL pp;
    GpiSetColor(hps, BACKGROUND(wip));
    MOVE(rct->xLeft+1, rct->yBottom+1);
    BOX(rct->xRight-2, rct->yTop-2);
}


PRIVAT VOID
ScrollSetSize(VIPINFO *wip)
{
    if (SCROLLDATA(wip, flags)&SVERT)
    {
        /* Calculate position and size of various parts */
        DOWNBUT(wip).xLeft   = wip->border;
        DOWNBUT(wip).yBottom = wip->border;
        DOWNBUT(wip).xRight  = wip->cx - wip->border-1;
        DOWNBUT(wip).yTop    = SCROLSIZE + wip->border;

        UPBUT(wip).xLeft   = DOWNBUT(wip).xLeft;
        UPBUT(wip).xRight  = DOWNBUT(wip).xRight;
        UPBUT(wip).yTop    = wip->cy - wip->border;
        UPBUT(wip).yBottom = wip->cy - SCROLSIZE - wip->border;

        MOVEBUT(wip).xLeft  = DOWNBUT(wip).xLeft;
        MOVEBUT(wip).xRight = DOWNBUT(wip).xRight;
    }
    if (SCROLLDATA(wip, flags)&SHORZ)
    {
        LEFTBUT(wip).xLeft = wip->border;
        LEFTBUT(wip).yBottom = wip->border;
        LEFTBUT(wip).xRight = SCROLSIZE + wip->border;
        LEFTBUT(wip).yTop = wip->cy - wip->border;

        RIGHTBUT(wip).yBottom = LEFTBUT(wip).yBottom;
        RIGHTBUT(wip).yTop = LEFTBUT(wip).yTop;
        RIGHTBUT(wip).xRight = wip->cx - wip->border;
        RIGHTBUT(wip).xLeft = wip->cx - SCROLSIZE - wip->border;

        MOVEBUT(wip).yBottom = LEFTBUT(wip).yBottom;
        MOVEBUT(wip).yTop = LEFTBUT(wip).yTop;
    }
    ScrollSetMove(wip);
}


PRIVAT VOID
ScrollSetMove(VIPINFO *wip)
{
    if (SCROLLDATA(wip, flags)&SVERT)
    {
        LONG s, sl;

        /* Calculate space for moving slider */
        s = UPBUT(wip).yBottom - DOWNBUT(wip).yTop + 1;
        /* Calculate size of moving slider */
        sl = (s * SCROLLDATA(wip, size))/SCROLLDATA(wip,range) ;
        /* Calculate offset */
        MOVEBUT(wip).yTop = UPBUT(wip).yBottom - ((s-sl) * SCROLLDATA(wip, current))/SCROLLDATA(wip,range);
        MOVEBUT(wip).yBottom = MOVEBUT(wip).yTop - sl;
    }
    if (SCROLLDATA(wip, flags)&SHORZ)
    {
        LONG s, sl;
        /* Calculate space for moving slider */
        s = RIGHTBUT(wip).xLeft - LEFTBUT(wip).xRight + 1;
        /* Calculate size of moving slider */
        sl = (s * SCROLLDATA(wip, size))/SCROLLDATA(wip,range) ;
        /* Calculate offset */
        MOVEBUT(wip).xLeft = LEFTBUT(wip).xRight + ((s-sl) * SCROLLDATA(wip, current))/SCROLLDATA(wip,range);
        MOVEBUT(wip).xRight = MOVEBUT(wip).xLeft + sl;
    }
}
