
/*
 * dosh.h:
 *      header file for dosh.c.
 *      dosh.c contains Control Program helper functions that are
 *      independent of a single application, i.e. you can use them in
 *      any program.
 *
 *      Function prefixes (new with V0.81):
 *      --  dosh*   Dos (Control Program) helper functions
 *
 *      Copyright (C) 1997-99 Ulrich Mller.
 *      This file is part of the XFolder source package.
 *      XFolder is free software; you can redistribute it and/or modify
 *      it under the terms of the GNU General Public License as published
 *      by the Free Software Foundation, in version 2 as it comes in the
 *      "COPYING" file of the XFolder main distribution.
 *      This program is distributed in the hope that it will be useful,
 *      but WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *      GNU General Public License for more details.
 */

#if __cplusplus
extern "C" {
#endif

#ifndef DOSH_HEADER_INCLUDED
    #define DOSH_HEADER_INCLUDED

    /*
     * doshGetULongTime:
     *      this returns the current time as a ULONG value (in milliseconds).
     *      Useful for stopping how much time the machine has spent in
     *      a certain function. To do this, simply call this function twice,
     *      and subtract the two values, which will give you the execution
     *      time in milliseconds.
     */

    ULONG doshGetULongTime(VOID);

    /*
     * doshQueryShiftState:
     *      returns TRUE if any of the SHIFT keys are
     *      currently pressed
     */

    BOOL doshQueryShiftState(VOID);

    /*
     * doshIsWarp4:
     *      returns TRUE if at least OS/2 Warp 4 is running, FALSE otherwise
     */

    BOOL doshIsWarp4(VOID);

    /*
     * doshEnumDrives:
     *      this copies the drive letters of the drives on the
     *      system to pszBuffer which match the pszFileSystem
     *      string (for example, "HPFS"); if pszFileSystem
     *      is NULL, all drives are enumerated.
     *      pszBuffer should be 30 characters in size.
     */

    VOID doshEnumDrives(PSZ pszBuffer, PSZ pszFileSystem);

    /*
     * doshQueryBootDrive:
     *      returns the letter of the boot drive
     */

    CHAR doshQueryBootDrive(VOID);

    /*
     * doshIsFixedDisk:
     *      checks whether a disk is fixed or removeable.
     *      ulLogicalDrive must be 1 for drive A:, 2 for B:, ...
     *      The result is stored in *pfFixed.
     *      Returns DOS error code.
     */

    APIRET doshIsFixedDisk(ULONG  ulLogicalDrive,        // in: 1 = A:, 2 = B:, ...
                             PBOOL  pfFixed);            // out: TRUE for fixed disks

    /*
     * doshAssertDrive:
     *      this checks for whether the given drive
     *      is currently available. If fSuppress == TRUE,
     *      DosError is used in a critical section to
     *      avoid "Drive not ready" popups.
     */

    APIRET doshAssertDrive(ULONG ulLogicalDrive, BOOL fSuppress);

    /*
     * doshQueryDiskFree:
     *       returns the number of bytes remaining on the disk
     *       specified by the given logical drive, or -1 upon errors.
     *       Note: This returns a "double" value, because a ULONG
     *       can only hold values of some 4 billion, which would
     *       lead to funny results for drives > 4 GB.
     */

    double doshQueryDiskFree(ULONG ulLogicalDrive); // 1 for A:, 2 for B:, 3 for C:, ...

    /*
     * doshQueryDiskFSType:
     *       copies the file-system type of the given disk object
     *       (HPFS, FAT, CDFS etc.) fo pszBuf.
     *       Returns the DOS error code.
     */

    APIRET doshQueryDiskFSType(ULONG ulLogicalDrive, // in:  1 for A:, 2 for B:, 3 for C:, ...
                               PSZ   pszBuf);        // out: buffer for FS type

    /*
     * doshQueryDiskType:
     *      this retrieves more information about a given drive,
     *      which is stored in *pulDeviceType. According to the
     *      IBM Control Program Reference, this value can be:
     *          0  48 TPI low-density diskette drive
     *          1  96 TPI high-density diskette drive
     *          2  3.5-inch 720KB diskette drive
     *          3  8-Inch single-density diskette drive
     *          4  8-Inch double-density diskette drive
     *          5  Fixed disk
     *          6  Tape drive
     *          7  Other (includes 1.44MB 3.5-inch diskette drive)
     *          8  R/W optical disk
     *          9  3.5-inch 4.0MB diskette drive (2.88MB formatted)
     *      Returns DOS error code.
     */

    APIRET doshQueryDiskType(ULONG  ulLogicalDrive,      // in: 1 = A:, 2 = B:, ...
                             PULONG pulDeviceType);      // out: device type

    /*
     * doshIsFileOnFAT:
     *      returns TRUE if pszFileName resides on
     *      a FAT drive
     */

    BOOL doshIsFileOnFAT(PSZ pszFileName);

    /*
     * doshIsValidFileName:
     *      this returns NO_ERROR only if pszFile is a valid file name.
     *      This may include a full path.
     *      If a drive letter is specified, this checks for whether
     *      that drive is a FAT drive and adjust the checks accordingly,
     *      i.e. 8+3 syntax and more invalid characters.
     *      If no drive letter is specified, this check is performed
     *      for the current drive.
     *      Note: this performs syntactic checks only. This does not
     *      check for whether the specified path components exist.
     *      However, it _is_ checked for whether the given drive
     *      exists.
     *      If an error is found, the corresponding DOS error code
     *      is returned:
     *          ERROR_INVALID_DRIVE
     *          ERROR_FILENAME_EXCED_RANGE  (on FAT: no 8+3 filename)
     *          ERROR_INVALID_NAME          (invalid character)
     */

    APIRET doshIsValidFileName(PSZ pszFile);

    /*
     * doshMakeRealName:
     *      this copies pszSource to pszTarget, replacing
     *      all characters which are not supported by file
     *      systems with cReplace.
     *      pszTarget must be at least the same size as pszSource.
     *      If (fIsFAT), the file name will be made FAT-compliant (8+3).
     *      Returns TRUE if characters were replaced.
     */

    BOOL doshMakeRealName(PSZ pszTarget, PSZ pszSource, CHAR cReplace, BOOL fIsFAT);

    /*
     * doshQueryFileSize:
     *      returns the size of an already opened file
     */

    ULONG doshQueryFileSize(HFILE hFile);

    /*
     * doshQueryPathSize:
     *      returns the size of any file
     */

    ULONG doshQueryPathSize(PSZ pszFile);

    /*
     * doshQueryDirExist:
     *      returns TRUE if the given directory
     *      exists.
     */

    BOOL doshQueryDirExist(PSZ pszDir);

    /*
     * doshCreatePath:
     *      this creates the specified directory.
     *      As opposed to DosCreateDir, this
     *      function can create several directories
     *      at the same time, if the parent
     *      directories do not exist yet.
     */

    APIRET doshCreatePath(PSZ pszPath);

    /*
     * doshSetCurrentDir:
     *      sets the current working directory
     *      to the given path.
     *      As opposed to DosSetCurrentDir, this
     *      one will change the current drive
     *      also, if one is specified.
     */

    APIRET doshSetCurrentDir(PSZ pszDir);

    /*
     * doshReadTextFile:
     *      reads a text file from disk, allocates memory
     *      via malloc() and returns a pointer to this
     *      buffer (or NULL upon errors). Specify in
     *      ulExtraMemory how much extra memory (in addition
     *      to the file's size) should be allocated to
     *      allow text manipulation.
     *      Returns NULL if an error occured.
     */

    PSZ doshReadTextFile(PSZ pszFile, ULONG ulExtraMemory);

    /*
     * doshCreateBackupFileName:
     *      creates a valid backup filename of pszExisting
     *      with a numerical file name extension which does
     *      not exist in the directory where pszExisting
     *      resides.
     *      Returns a PSZ to a buffer which was allocated
     *      using malloc().
     */

    PSZ doshCreateBackupFileName(PSZ pszExisting);

    /*
     * doshWriteTextFile:
     *      writes a text file to disk; pszFile must contain the
     *      whole path and filename.
     *      An existing file will be backed up if (fBackup).
     *      Returns the number of bytes written.
     */

    ULONG doshWriteTextFile(PSZ pszFile, PSZ pszContent, BOOL fBackup);

    /*
     * doshOpenLogFile:
     *      this opens a log file in the root directory of
     *      the boot drive; it is titled pszFilename, and
     *      the file handle is returned.
     */

    HFILE doshOpenLogFile(PSZ pszFileName);

    /*
     * doshWriteToLogFile
     *      writes a string to a log file.
     */

    APIRET doshWriteToLogFile(HFILE hfLog, PSZ psz);

    /*
     * doshQuickStartSession:
     *      shortcut to DosStartSession w/out having to deal
     *      with all the parameters. This one starts a session
     *      as a child session and can optionally wait for the
     *      session to end (using a termination queue), if
     *      Wait == TRUE; otherwise, this function returns
     *      immediately.
     */

    APIRET doshQuickStartSession(PSZ pszPath,     // in: program to start
                                 PSZ pszParams,   // in: parameters for program
                                 BOOL Visible,    // in: show program?
                                 BOOL Wait,       // in: wait for termination?
                                 PULONG pulSID,   // out: session ID
                                 PPID ppid);      // out: process ID

    /*
     * doshSetPathAttr:
     *      sets the file attributes of pszFile,
     *      which must be fully qualified.
     *      fAttr can be:
     *              FILE_ARCHIVED
     *              FILE_READONLY
     *              FILE_SYSTEM
     *              FILE_HIDDEN
     */

    APIRET doshSetPathAttr(PSZ pszFile, ULONG fAttr);

#endif

#if __cplusplus
}
#endif

